;==========================================================================
;Version history:
;==========================================================================
;2010/01/14     First edition
;2010/12/01     Check if any single note/drum is played before turning off DAC in "F_SpuIrqSpeechService"
;==========================================================================
        .INCLUDE        Inc\Speech.inc
;==========================================================================
        .CODE
;==========================================================================
T_SpeechFreqTable:
        %LoadSpeechFreq         8000    ;Index=0
        %LoadSpeechFreq         9000    ;Index=1
        %LoadSpeechFreq         10000   ;Index=2
        %LoadSpeechFreq         12000   ;Index=3
        %LoadSpeechFreq         14000   ;Index=4
        %LoadSpeechFreq         16000   ;Index=5
        %LoadSpeechFreq         18000   ;Index=6
        %LoadSpeechFreq         20000   ;Index=7
        %LoadSpeechFreq         22000   ;Index=8
        %LoadSpeechFreq         24000   ;Index=9
;==========================================================================
;Purpose: Play a speech with a SPU channel
;Description: Use a SPU channel to play a speech
;Input: A = Speech index; X = Sampling rate index; Y = Channel number (GPCD6: 0 ~ 3; GPCD9: 0 ~ 7)
;Destroy: A, X, Y
;==========================================================================
F_PlayHWSpeech:                         ;Play a speech with SPU channel
        STA     R_SpeechIndex           ;Backup index
        STX     R_SpeechTemp            ;Index of the sampling rate
        CPY     #D_ChannelNo
        BCC     L_?
        RTS
L_?:
        .IF MIDI = ON
            .IF PITCH_BEND = ON
        LDA     #FFH
        STA     R_SpuMidiCh,Y
            .ENDIF
        .ENDIF

        LDA     P_DAC_Ctrl
        AND     #(D_DAC_Enable+D_OPN_En+D_OPP_En)
        CMP     #(D_DAC_Enable+D_OPN_En+D_OPP_En)
        BEQ     L_GOS?
        %TurnOnDAC
L_GOS?:
        LDA     R_SpeechIndex           ;Use the [R_SpeechIndex+1:R_SpeechIndex] to index 65536 speech files
        STA     R_SpeechDPTR            ;Backup index
        LDA     R_SpeechIndex+1
        STA     R_SpeechDPTR+1

        CLC
        ROL     R_SpeechIndex
        ROL     R_SpeechIndex+1

        CLC
        LDA     R_SpeechIndex
        ADC     R_SpeechDPTR
        STA     R_SpeechIndex
        LDA     R_SpeechIndex+1
        ADC     R_SpeechDPTR+1
        STA     R_SpeechIndex+1

        LDA     #.LOW.(T_SPEECH)        ;Get zero offset address LB
        STA     R_SpeechDPTR
        LDA     #.HIGH.(T_SPEECH)       ;Get zero offset address HB
        STA     R_SpeechDPTR+1

        CLC
        LDA     R_SpeechDPTR
        ADC     R_SpeechIndex           ;Add real index LB
        STA     R_SpeechDPTR
        LDA     R_SpeechDPTR+1
        ADC     R_SpeechIndex+1         ;Add real index HB
        STA     R_SpeechDPTR+1

        LDA     #00H
        STA     R_SpeechIndex+1         ;Clear speech index HB

        LDA     R_SpeechTemp            ;X = 0,1,2,...(Sampling rate)
        ASL     A
        TAX
        SEI
        LDA     T_SpeechFreqTable,X     ;Sampling rate LB
        STA     R_ChFsLTempS
        LDA     T_SpeechFreqTable+1,X   ;Sampling rate HB
        STA     R_ChFsHTempS
        CLI

        LDX     T_ChMap,Y
        STY     R_SpeechTemp            ;Backup channel

        LDA     P_INT_CtrlH
        ORA     #D_SPUIntEn             ;Enable SPU IRQ
        STA     P_INT_CtrlH

        LDA     P_SPU_Enable
        AND     T_ChEnable,Y
        BEQ     L_SpPlayDirectly        ;If the channel is not enabled, play a speech directly
        LDA     P_SPU_Conc              ;If a bit of P_SPU_Conc was not cleared by SPU, that means SPU can't find Z.C.
        AND     T_ChEnable,Y
        BNE     L_CloseCh?
        JMP     L_ConcatenationS
L_CloseCh?:
        LDA     P_SPU_Enable
        AND     T_ChDisable,Y           ;Close P_SPU_Enable first then play speech directly to prevent speech be connected without Z.C.
        STA     P_SPU_Enable
L_SpPlayDirectly:
        LDY     #00H
        LDA     (R_SpeechDPTR),Y
        STA     R_Ch0DPTRL,X            ;Speech address LB
        STA     R_Ch0DPTNL,X
        INY
        LDA     (R_SpeechDPTR),Y
        STA     R_Ch0DPTRM,X            ;Speech address HB
        STA     R_Ch0DPTNM,X
        INY
        LDA     (R_SpeechDPTR),Y
        ORA     R_ChDPTRHTempS
        STA     R_Ch0DPTRH,X            ;Speech address Bank
        STA     R_Ch0DPTNH,X
        AND     #D_Bit7                 ;Bit7 = 0,ADPCM; Bit7 = 1,PCM
        BNE     L_PCM?
        SEI
        LDA     #00H
        STA     R_Ch0LpLenL,X           ;Initialize ADPCM's loop length LB
        STA     R_Ch0LpLenH,X           ;Initialize ADPCM's loop length HB
        CLI
L_PCM?:
        LDA     #00H
        STA     R_Ch0S1,X               ;Algorithm coefficient data
        STA     R_Ch0S2,X               ;Algorithm coefficient data
        STA     R_Ch0PhAddL,X
        STA     R_Ch0PhAddH,X

        SEI
        LDA     R_ChFsLTempS            ;Set Sampling rate
        STA     R_Ch0FsL,X
        LDA     R_ChFsHTempS
        STA     R_Ch0FsH,X
        CLI

        LDA     #7FH
        STA     R_Ch0VolCtl,X

        SEI
        LDY     R_SpeechTemp
        LDA     T_ChEnable,Y
        STA     P_SPU_Status            ;Clear channel IRQ status

        LDA     R_ChSpeechStatus,Y
        ORA     #D_SpeechPlaying        ;Set playing flag
        AND     #.NOT.(D_SpeechRepeat+D_SpeechPause)
        STA     R_ChSpeechStatus,Y

        LDA     P_SPU_IntCtrl
        ORA     T_ChEnable,Y            ;Enable channel IRQ
        STA     P_SPU_IntCtrl

        LDA     P_SPU_Enable
        ORA     T_ChEnable,Y
        STA     P_SPU_Enable            ;Turn on the channel
        CLI
        JMP     L_ExitPlayHWSpeech
L_ConcatenationS:
        LDY     #00H
        LDA     (R_SpeechDPTR),Y
        STA     R_Ch0DPTNL,X            ;Speech address LB
        INY
        LDA     (R_SpeechDPTR),Y
        STA     R_Ch0DPTNM,X            ;Speech address HB
        INY
        LDA     (R_SpeechDPTR),Y
        ORA     R_ChDPTRHTempS          ;Set speech format (ADPCM or PCM)
        STA     R_Ch0DPTNH,X            ;Speech address Bank

        LDA     #7FH
        STA     R_Ch0VolN,X             ;Next sample's volume

        LDY     R_SpeechTemp
        SEI
        LDA     T_ChEnable,Y
        STA     P_SPU_Status            ;Clear channel IRQ status

        LDA     R_ChSpeechStatus,Y
        ORA     #D_SpeechZCJump+D_SpeechPlaying
        AND     #.NOT.(D_SpeechRepeat+D_SpeechPause)
        STA     R_ChSpeechStatus,Y

        LDA     P_SPU_IntCtrl
        ORA     T_ChEnable,Y
        STA     P_SPU_IntCtrl

        LDA     P_SPU_Conc
        ORA     T_ChEnable,Y
        STA     P_SPU_Conc              ;Enable concatenate jump
        CLI
L_ExitPlayHWSpeech:
        LDA     R_SpeechStatus
        ORA     T_ChEnable,Y            ;Save which SPU channel is playing speech
        STA     R_SpeechStatus
        RTS
;==========================================================================
F_SpuIrqSpeechService:
        LDA     P_SPU_Status            ;Check if any interrupt is issued
        AND     P_SPU_IntCtrl
        BNE     L_PrecedeS
        RTS
L_PrecedeS:
        STA     R_SpeechIrqCh
        AND     #11110000B
        BEQ     L_LowNibbleS
        LDA     R_SpeechIrqCh
        AND     #00001111B
        BEQ     L_HighNibbleS
        JMP     L_LowNibbleS
L_HighNibbleS:
        LDA     R_SpeechIrqCh
        LSR     A
        LSR     A
        LSR     A
        LSR     A
        AND     #00001111B
        TAX
        LDA     T_Precede,X
        CLC
        ADC     #04H
        TAX                             ;Get the channel
        LDY     T_ChMap,X
        JSR     F_SetSpeechIrqPara
L_GoNextHighS:
        LDA     R_SpeechIrqCh
        AND     T_ChDisable,X
        STA     R_SpeechIrqCh
        AND     #11110000B
        BEQ     L_CheckTurnOff
        JMP     L_HighNibbleS
L_LowNibbleS:
        LDA     R_SpeechIrqCh
        AND     #00001111B
        TAX
        LDA     T_Precede,X
        TAX                             ;Get the channel
        LDY     T_ChMap,X
        JSR     F_SetSpeechIrqPara
L_GoNextLowS:
        LDA     R_SpeechIrqCh
        AND     T_ChDisable,X
        STA     R_SpeechIrqCh
        AND     #00001111B
        BNE     L_LowNibbleS

        LDA     R_SpeechIrqCh
        AND     #11110000B
        BEQ     L_CheckTurnOff
        JMP     L_HighNibbleS
L_ExitSpuIrqSpService:
        RTS

L_CheckTurnOff:
        LDA     R_SpeechStatus          ;Check if any speech is played with SPU
        BNE     L_ExitSpuIrqSpService

        .IF MIDI = ON
        LDA     R_MIDIStatus            ;Check if MIDI is playing
        AND     #D_MIDIEn
        BNE     L_ExitSpuIrqSpService

        LDA     R_SingleNote            ;Check if any single note is played
        ORA     R_SingleDrum            ;Check if any single drum is played
        BNE     L_ExitSpuIrqSpService
        .ENDIF

        .IF SW_ChA = ON
        %IsActiveChA                    ;Check if SW_ChA is active
        BCS     L_ExitSpuIrqSpService
        .ENDIF

        .IF SW_ChB = ON
        %IsActiveChB                    ;Check if SW_ChB is active
        BCS     L_ExitSpuIrqSpService
        .ENDIF

        .IF Comair5 = ON
        JSR     F_CA5_IsActive
        BCS     L_ExitSpuIrqSpService
        .ENDIF

        %TurnOffDAC
        RTS
;==========================================================================
F_SetSpeechIrqPara:
        LDA     R_ChSpeechStatus,X
        AND     #D_SpeechPlaying
        BNE     L_GONS?
        RTS
L_GONS?:
        LDA     R_ChSpeechStatus,X
        AND     #D_SpeechZCJump         ;Check if concatenate jump mode is enabled
        BNE     L_SpeechZCIrqJump?

        LDA     R_ChSpeechStatus,X
        AND     #D_SpeechRepeat         ;Check if speech repeat flag is set
        BNE     L_ChkRepeat
L_SpeechEnd?:
        LDA     R_ChSpeechStatus,X      ;Close channel when a speech is finished
        AND     #.NOT.(D_SpeechPlaying)
        STA     R_ChSpeechStatus,X

        LDA     P_SPU_IntCtrl
        AND     T_ChDisable,X
        STA     P_SPU_IntCtrl           ;Disable Channel IRQ

        LDA     P_SPU_Enable
        AND     T_ChDisable,X
        STA     P_SPU_Enable            ;Close Channel

        LDA     R_SpeechStatus
        AND     T_ChDisable,X
        STA     R_SpeechStatus
        RTS

L_SpeechZCIrqJump?:                     ;Set parameter if concatenate jump mode is set
        LDA     R_ChSpeechStatus,X
        ORA     #D_SpeechPlaying
        AND     #.NOT.(D_SpeechZCJump)
        STA     R_ChSpeechStatus,X

        LDA     R_ChFsLTempS            ;Set sampling rate
        STA     R_Ch0FsL,Y
        LDA     R_ChFsHTempS
        STA     R_Ch0FsH,Y

        LDA     T_ChEnable,X
        STA     P_SPU_Status
        RTS

L_ChkRepeat:                            ;Set parameter if repeat flag is set
        .IF REPEAT_SPEECH = ON
        DEC     R_RepeatCounter
        LDA     R_RepeatCounter
        CMP     #01H
        BEQ     L_DisableRepeat
        BCC     L_CloseRepeat?
        LDA     T_ChEnable,X
        STA     P_SPU_Status            ;Clear interrupt flag
        RTS

L_CloseRepeat?:
        LDA     P_SPU_Enable            ;Repeat counter = 0, close the corresponding channel
        AND     T_ChDisable,X
        STA     P_SPU_Enable            ;Close SPU

        LDA     P_SPU_IntCtrl
        AND     T_ChDisable,X
        STA     P_SPU_IntCtrl           ;Disable IRQ

        LDA     R_ChSpeechStatus,X
        AND     #.NOT.(D_SpeechPlaying+D_SpeechRepeat)
        STA     R_ChSpeechStatus,X

        LDA     T_ChEnable,X
        STA     P_SPU_Status

        LDA     R_SpeechStatus
        AND     T_ChDisable,X
        STA     R_SpeechStatus
        RTS

L_DisableRepeat:
        LDA     T_ChEnable,X            ;If repeat counter = 1, clear the repeat flag
        STA     P_SPU_Status

        LDA     R_Ch0VolCtl,Y
        AND     #01111111B
        STA     R_Ch0VolCtl,Y           ;Set back to Loop mode
        RTS
        .ENDIF

        LDA     P_SPU_IntCtrl
        AND     T_ChDisable,X
        STA     P_SPU_IntCtrl

        LDA     T_ChEnable,X
        STA     P_SPU_Status
        RTS
;==========================================================================
;Purpose: Increase the volume of speech on designated channel with 9 levels for SPU
;Level: 00, 16, 32, 48, 64, 80, 96, 112, 127
;Input: Y = Channel number (GPCD6: 0 ~ 3; GPCD9: 0 ~ 7)
;Destroy: A, X, Y
;==========================================================================
F_SetSpChVolUp:
        CPY     #D_ChannelNo
        BCC     L_?
        RTS
L_?:
        TYA
        TAX
        LDY     T_ChMap,X

        LDA     R_Ch0VolCtl,Y
        AND     #01111111B
        CLC
        ADC     #10H                    ;Plus 10H to volume per time
        STA     R_SpeechTemp

        CMP     #7FH
        BCS     L_SetToMaxSpVol

        SEI
        LDA     R_Ch0VolCtl,Y
        AND     #10000000B
        ORA     R_SpeechTemp
        STA     R_Ch0VolCtl,Y
        STA     R_Ch0VolN,Y
        CLI
        JMP     L_ExitSetSpChVolUp

L_SetToMaxSpVol:
        SEI                             ;If volume >= 7FH, set the volume to 7FH
        LDA     R_Ch0VolCtl,Y
        ORA     #7FH
        STA     R_Ch0VolCtl,Y
        STA     R_Ch0VolN,Y
        CLI
L_ExitSetSpChVolUp:
        RTS
;==========================================================================
;Purpose: Decrease the volume of speech on designated channel with 9 levels for SPU
;Level: 127, 112, 96, 80, 64, 48, 32, 16, 00
;Input: Y = Channel number (GPCD6: 0 ~ 3; GPCD9: 0 ~ 7)
;Destroy: A, X, Y
;==========================================================================
F_SetSpChVolDn:
        CPY     #D_ChannelNo
        BCC     L_?
        RTS
L_?:
        TYA
        TAX
        LDY     T_ChMap,X

        LDA     R_Ch0VolCtl,Y
        AND     #01111111B
        CMP     #7FH
        BEQ     L_SetToVol
        SEC
        SBC     #10H
        STA     R_SpeechTemp

        CMP     #00H
        BEQ     L_SetToMinSpVol
        BMI     L_SetToMinSpVol

        SEI
        LDA     R_Ch0VolCtl,Y
        AND     #10000000B
        ORA     R_SpeechTemp
        STA     R_Ch0VolCtl,Y
        STA     R_Ch0VolN,Y
        CLI
        JMP     L_ExitSetSpChVolUp

L_SetToVol:
        SEI
        LDA     R_Ch0VolCtl,Y
        AND     #10000000B
        ORA     #70H
        STA     R_Ch0VolCtl,Y
        STA     R_Ch0VolN,Y
        CLI
        JMP     L_ExitSetSpChVolUp

L_SetToMinSpVol:
        SEI
        LDA     R_Ch0VolCtl,Y
        AND     #10000000B
        STA     R_Ch0VolCtl,Y
        STA     R_Ch0VolN,Y
        CLI
L_ExitSetSpChVolDn:
        RTS
;==========================================================================
T_Precede:                              ;Table for precede
        DB      00H
        DB      00H
        DB      01H
        DB      00H
        DB      02H
        DB      00H
        DB      01H
        DB      00H
        DB      03H
        DB      00H
        DB      01H
        DB      00H
        DB      02H
        DB      00H
        DB      01H
        DB      00H
;==========================================================================
